module slidegame


//	**************************************************************************************************
//
//	A simple slide game that uses bitmaps to show nice pictures.
//	On Macintosh one should select a PICT file; on Windows(95/NT) one should select a BMP file.
//
//	The program has been written in Clean 1.3.1 and uses the Clean Standard Object I/O library 1.1
//	
//	**************************************************************************************************


import StdEnv, StdIO, Random


/*	Start simply creates the slide game process.
	Note that the slide game process is polymorphic in the local and public process state.
	Because we need to choose a value for these states we use the singleton type NoState.
*/
::	NoState							// A dummy state
	=	NoState

Start :: *World -> *World
Start world
	=	startIO NoState NoState [openSlideGame] [] world


/*	openSlideGame first attempts to read in the bitmap.
	If successfull, openSlideGame then checks whether the given bitmap has proper dimensions.
	If this is the case then a window is opened that will contain the slide game.
	The initial positions of the slides are determined by shuffling them nr_shuffle times randomly. 
	The local state of the window keeps track of the current position of the hole.
*/
::	WindowState
	=	{	curHole	:: Coord		// The current position of the hole
		}
::	Coord
	=	{	col		:: Int			// The zero based column number
		,	row		:: Int			// The zero based row    number
		}

openSlideGame :: (PSt .l .p) -> PSt .l .p
openSlideGame ps
	#	(maybeFile,ps)	= selectInputFile ps
	|	isNothing maybeFile
		=	closeProcess ps
	#	(maybeBitmap,ps)= accFiles (openBitmap (fromJust maybeFile)) ps
	|	isNothing maybeBitmap
		=	closeProcess ps
	#	bitmap			= fromJust maybeBitmap
		bitmapsize		= getBitmapSize bitmap
		blocksize		= {w=bitmapsize.w/4,h=bitmapsize.h/4}
	|	not (ok_blocksize blocksize)
		=	closeProcess ps
	#	(seed,ps)		= getNewRandomSeed ps
		(okCoords,hole)	= initlast [{col=col,row=row} \\ row<-[0..3],col<-[0..3]]
		(_,coords,hole)	= iteraten nr_shuffle shuffle (seed,zip2 okCoords okCoords,hole)
	#	(windowId,ps)	= accPIO openId ps
	#	(allcids, ps)	= accPIO (openIds   15) ps
	#	(allr2ids,ps)	= accPIO (openR2Ids 15) ps
		wdef			= window bitmap blocksize windowId allcids allr2ids coords
	#	(error,ps)		= openWindow {curHole=hole} wdef ps
	|	error==NoError
		=	ps
	|	otherwise
		=	closeProcess ps
where
	nr_shuffle			= 200
	
	ok_blocksize {w,h}
		=	isbetween minblocksize.w w maxblocksize.w && isbetween minblocksize.h h maxblocksize.h
	where
		minblocksize	= {w=20,h=20}
		maxblocksize	= {w=maxFixedWindowSize.w/4,h=maxFixedWindowSize.h/4}
	
	shuffle :: (RandomSeed,[(Coord,Coord)],Coord) -> (RandomSeed,[(Coord,Coord)],Coord)
	shuffle (seed,coords,hole)
		#	(candidates,others)	= splitFilter (\(okCoord,coord)->distCoord coord hole==1) coords
			(random_nr,seed)	= random seed
			(before,[(okCandidate,candidate):after])
		  						= splitAt (random_nr mod (length candidates)) candidates
		=	(seed,before++[(okCandidate,hole):after]++others,candidate)


/*	window defines the Window that shows the slide game.
	It contains a list of slide controls.
	Closing the window will terminate the program.
*/
window :: Bitmap Size Id [Id] [SlideR2Id] [(Coord,Coord)]
	-> Window (ListLS (AddLS (:+: CustomButtonControl (Receiver2 SlideMsgIn Bool)))) WindowState (PSt .l .p)
window bitmap blocksize windowId allcids allr2ids coords
	=	Window "SlideGame"
			(	ListLS (map (slideControl bitmap blocksize windowId allr2ids) coord_ids)
			)
			[	WindowClose 	(noLS closeProcess)
			,	WindowId		windowId
			,	WindowItemSpace 0 0
			,	WindowSize		{w=4*blocksize.w,h=4*blocksize.h}
			,	WindowLook		(\_ {newFrame}-> seq [setPenColour Grey,fill newFrame])
			]
where
	coord_ids	= zip2 coords (zip2 allcids allr2ids)


/*	slideControl defines one slide control of the slide game.
	A slide control consists of two components:
	*	A custom button control:
			This control shows a part of the bitmap image.
			Selecting this control will swap places with the current hole iff it is adjacent to the 
			hole.
			It checks whether all slide controls are at their desired locations, and if so disables 
			the window.
			Note that disabling the window will disable all slide controls. The look of a slide control 
			is such that in disabled state it will not frame its bitmap part, so the complete bitmap 
			will be displayed.
	*	A receiver control:
			This control handles external requests that inform whether the slide control is at its 
			desired position.
*/
::	SlideState						// The local state of a slide control
	=	{	curCoord :: Coord		// The current location of the slide control
		}
::	SlideMsgIn						// The ingoing messages of the slide control
	=	AreYouOk					// Inform whether the control is currently at its desired location
::	SlideMsgOut						// The outgoing messages of the slide control
	:==	Bool						// True iff the control is currently at its desired location
::	SlideR2Id						// Shorthand for the receiver id of a slide control
	:==	R2Id SlideMsgIn SlideMsgOut
::	SlideControl ls ps				// Shorthand for the slide control constructor type
	:==	AddLS (:+: CustomButtonControl (Receiver2 SlideMsgIn SlideMsgOut)) ls ps

slideControl :: Bitmap Size Id [SlideR2Id] ((Coord,Coord),(Id,SlideR2Id))
	-> SlideControl WindowState (PSt .l .p)
slideControl bitmap size windowId allr2ids ((okCoord,initCoord),(cid,r2id))
	=	{	addLS	= { curCoord=initCoord }
		,	addDef	= custombutton :+: receiver2
		}
where
	others			= removeMember r2id allr2ids
	
	custombutton	= CustomButtonControl size slideLook
						[	ControlPos		(LeftTop,offset initCoord)
						,	ControlFunction slideMove
						,	ControlId		cid
						]
	slideLook select {newFrame}
					= seq [	drawAt {x=0-okCoord.col*size.w,y=0-okCoord.row*size.h} bitmap
						  :	if (enabled select) [draw newFrame] []
						  ]
	offset {col,row}= {vx=size.w*col,vy=size.h*row}
	
	slideMove :: (.(SlideState,WindowState),PSt .l .p) -> (.(SlideState,WindowState),PSt .l .p)
	slideMove ((slide=:{curCoord},ls=:{curHole}),ps)
		|	distCoord curCoord curHole<>1
			=	((slide,ls),ps)
		#	slide			= {slide & curCoord=curHole }
			ls				= {ls    & curHole =curCoord}
		#	(_,ps)			= accPIO (setControlPos windowId cid (LeftTop,offset curHole)) ps
		#	i_am_ok			= curHole==okCoord
		|	not i_am_ok
			=	((slide,ls),ps)
		#	(others_ok,ps)	= seqList (map areYouOk others) ps
		|	and others_ok
			=	((slide,ls),appPIO (disableWindow windowId) ps)
		|	otherwise
			=	((slide,ls),ps)
	
	areYouOk :: SlideR2Id (PSt .l .p) -> (Bool,PSt .l .p)
	areYouOk r2id ps
		#	(response,ps)	= syncSend2 r2id AreYouOk ps
		=	(fromJust (snd response),ps)
	
	receiver2	= Receiver2 r2id receive2 []
	
	receive2 :: SlideMsgIn ((SlideState,.ls),PSt .l .p) -> (SlideMsgOut,((SlideState,.ls),PSt .l .p))
	receive2 AreYouOk (slide=:({curCoord},_),ps)
		=	(okCoord==curCoord,(slide,ps))

//	The distance between two Coords:
distCoord :: !Coord !Coord -> Int
distCoord coord1 coord2
	=	abs (coord1.col-coord2.col) + abs (coord1.row-coord2.row)

instance zero Coord where
	zero = {col=0,row=0}
instance == Coord where
	(==) coord1 coord2 = distCoord coord1 coord2==0

//	Generally useful functions:
initlast :: ![.a] -> ([.a],.a)							// (init xs,last xs)
initlast [a]
	=	([],a)
initlast [a:as]
	=	([a:init],last)
where
	(init,last)	= initlast as
initlast []
	=	abort "initlast of []"

splitFilter :: (a -> .Bool) !.[a] -> (.[a],.[a])		// (filter cond xs,filter (not o cond) xs)
splitFilter f [a:as]
	|	f a
		=	([a:yes],no)
	|	otherwise
		=	(yes,[a:no])
where
	(yes,no)	= splitFilter f as
splitFilter f []
	=	([],[])

iteraten :: !Int (a -> a) a -> a
iteraten n f x
	|	n<=0
		=	x
	|	otherwise
		=	iteraten (n-1) f (f x)

isbetween :: !a !a a -> Bool	| Ord a
isbetween low x up
	=	low<=x && x<=up
